<?php

namespace Database\Seeders;

use App\Models\App\JobPost\ExperienceLevel;
use App\Models\App\JobPost\JobPost;
use Database\Seeders\App\JobPost\ExperienceLevelSeeder;
use Database\Seeders\Traits\DisableForeignKeys;
use Database\Seeders\Traits\TruncateTable;
use Exception;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Ramsey\Uuid\Uuid;

class AppVersionUpdateSeeder extends Seeder
{
    use TruncateTable, DisableForeignKeys;

    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {

        $this->call(ExperienceLevelSeeder::class);
        $this->insertExperienceLevel();
        $this->insertWorkArrangement();
        $this->jobPostApplyFormConversion();
    }

    public function insertWorkArrangement(): void
    {
        JobPost::query()->update(['working_arrangement' => 'on_site']);
    }

    public function insertExperienceLevel(): void
    {
        $experienceLevel = ExperienceLevel::query()->first();

        JobPost::query()->update(['experience_level_id' => $experienceLevel->id]);

    }

    public function jobPostApplyFormConversion()
    {
        DB::beginTransaction();

        try {
            JobPost::query()
                ->select('id', 'apply_form_settings')
                ->get()
                ->map(function ($record) {
                    return $this->transformData($record);
                });

            DB::commit();
            return "Transaction successful";
        } catch (Exception $e) {
            DB::rollback();
            return $e;
        }
    }

    public function transformData($record)
    {
        $this->newData = [];
        $this->prepareBasicInformation();

        $oldRecord = gettype($record->apply_form_settings) === 'string' ? json_decode($record->apply_form_settings) : $record->apply_form_settings;

        foreach ($oldRecord as $item) {

            if (is_string($item)) {
                $item = json_decode($item);
            }

            $newItem = [
                'id' => Uuid::uuid4()->toString(),
                'is_visible' => true,
                'title' => $item->title ?? '',
                'key' => str_replace(' ', '_', strtolower($item->title)),
                'actions' => [
                    "edit" => true,
                    "delete" => false,
                    "move" => false
                ],
                'items' => $this->transformFields($item->fields),
            ];

            $this->newData[] = $newItem;
        }

        $record->apply_form_settings = json_encode($this->newData);
        $record->save();

        return $this;
    }

    public function transformFields($fields): array
    {
        return [
            [
                'actions' => [
                    "edit" => true,
                    "delete" => true,
                    "move" => true
                ],
                'fields' => $this->transformFieldItems($fields),
                'id' => Uuid::uuid4()->toString(),
            ]
        ];
    }

    public function transformFieldItems($fields): array
    {
        $newFields = [];
        foreach ($fields as $field) {
            $newField = [
                'actions' => [
                    "edit" => true,
                    "delete" => true,
                    "move" => true
                ],
                'is_visible' => $field->show,
                'required' => $field->require,
                'title' => $field->name,
                'type' => $field->type,
                'options' => [],
                'id' => Uuid::uuid4()->toString(),
                'fields' => [],
            ];

            $newFields[] = $newField;
        }

        return $newFields;
    }

    public function prepareBasicInformation(): array
    {

        return $this->newData[0] = [
            'actions' => [
                'delete' => false,
                'edit' => true,
                'move' => false,

            ],
            'id' => "64adc53a-4cdf-4c4e-82e5-b3bfc6b990f9",
            'is_visible' => true,
            'items' => [
                'actions' => [
                    'delete' => false,
                    'edit' => true,
                    'move' => false,
                ],
                'fields' => $this->getBasicInformationFields(),
                'id' => "dac64396-e2a6-4c1b-bf30-ad864f3c7ff0"
            ],
            'key' => "basic_information",
            'title' => "Basic Information"

        ];
    }

    public function getBasicInformationFields(): array
    {
        $fields = [
            'First name' => [
                'type' => 'text',
                'required' => true,
                'options' => []
            ],
            'Last name' => [
                'type' => 'text',
                'required' => true,
                'options' => []
            ],
            'Email' => [
                'type' => 'email',
                'required' => true,
                'options' => []
            ],
            'Gender' => [
                'type' => 'radio',
                'required' => false,
                'options' => [
                    'Male', 'Female'
                ]
            ],
            'Date of birth' => [
                'type' => 'date',
                'required' => false,
                'options' => []
            ],
            'Phone' => [
                'type' => 'tel-input',
                'required' => false,
                'options' => []
            ],
        ];

        $insertArray = [];

        foreach ($fields as $key => $field) {
            $insertArray[] = [
                'actions' => [
                    'delete' => false,
                    'edit' => false,
                    'move' => false,
                ],
                'is_visible' => true,
                'options' => is_array($field) ? $field['options'] : $field,
                'required' => is_array($field) ? $field['required'] : $field,
                'title' => $key,
                'type' => is_array($field) ? $field['type'] : $field,
            ];
        }

        return $insertArray;
    }
}
